
// SimParm: Simple and flexible C++ configuration framework
// Copyright (C) 2007 Australian National University
// 
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2 of the License, or (at your option) any later version.
// 
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
// 
// Contact:
// Kevin Pulo
// kevin.pulo@anu.edu.au
// Leonard Huxley Bldg 56
// Australian National University, ACT, 0200, Australia

//package simparm.twiddler;

import java.util.*;
import java.io.*;


class ConfigSet {

	public Vector<ConfigEntry> ordered;
	public Map<String,Integer> lookup;

	public InputStream in;
	public PrintStream out;

	public ConfigSet() {
		ordered = new Vector<ConfigEntry>();
		lookup = new HashMap<String,Integer>();
		in = System.in;
		out = System.out;
	}

	public ConfigSet(InputStream in, PrintStream out) {
		this();
		this.in = in;
		this.out = out;
	}

	public void add(ConfigEntry entry) {
		ordered.add(entry);
		lookup.put(entry.getName(), ordered.size() - 1);
	}

	public ConfigEntry get(String name) {
		return ordered.elementAt(lookup.get(name).intValue());
	}


	//public void outputDefinition(ostream &out);

	private void readEntry(BufferedReader input, Twiddler twiddler) throws IOException {
		String type = ConfigEntry.readString(input, "type");

		ConfigEntry entry;
		if (type.equals("string")) {
			entry = new ConfigEntryString();
		} else if (type.equals("choice")) {
			entry = new ConfigEntryChoice();
		} else if (type.equals("double")) {
			entry = new ConfigEntryDouble();
		} else if (type.equals("unsigned_long")) {
			//entry = new ConfigEntryUnsignedLong();
			entry = new ConfigEntryLong(true);
		} else if (type.equals("long")) {
			entry = new ConfigEntryLong();
		} else if (type.equals("bool")) {
			entry = new ConfigEntryBool();
		} else if (type.equals("trigger")) {
			entry = new ConfigEntryTrigger();
		} else if (type.equals("divider")) {
			entry = new ConfigEntryDivider();
		} else {
			System.err.println("Unexpected type \"" + type + "\"");
			System.exit(1);
			return;
		}
		entry.setOutput(out);
		entry.readDefinition(input);
		entry.addToTwiddler(twiddler);
		add(entry);
	}

	public void inputDefinition(Twiddler twiddler) {
		try {
			BufferedReader input = new BufferedReader(new InputStreamReader(in));
			String s = input.readLine();
			while (s.equals("pulse")) {
				s = input.readLine();
			}
			int numEntries = Integer.parseInt(s);
			System.err.println("numEntries = " + numEntries);
			for (int i = 0; i < numEntries; i++) {
				readEntry(input, twiddler);
			}
		} catch (IOException e) {
			System.err.println("exception " + e );
		}

	}

	public boolean inputSingleEntry(String[] word, int from) {
		if (word.length <= from) {
			System.err.println("Error reading config entry: expected name, but line ended prematurely");
			return false;
		}

		String name = word[from];
		if (name.charAt(0) == '#') {
			// comment, so ignore
			return true;
		}

		if (word.length <= from + 1) {
			System.err.println("Error reading config entry: expected '=', but line ended prematurely");
			return false;
		}
		if ( ! word[from + 1].equals("=")) {
			System.err.println("Error reading config entry: expected '=', found \"" + word[from + 1] + "\"");
			return false;
		}

		if ( ! lookup.containsKey(name)) {
			System.err.println("Error reading config entry: unknown parameter name \"" + name + "\"");
			return false;
		}

		ConfigEntry entry = ordered.elementAt(lookup.get(name));
		if (word.length <= from + 2) {
			System.err.println("Error reading config entry: expected value for \"" + name + "\", but line ended prematurely");
			return false;
		}
		String value = word[from + 2];
		if ( ! entry.setValueFromInputStream(value)) {
			System.err.println("Error reading config entry: error setting value for parameter \"" + name + "\" to \"" + value + "\"");
			return false;
		}

		//System.err.println("Set " + name + " = " + value);
		return true;
	}

	//public void inputAllEntries(BufferedReader input) {
	//}

	private void readUpdate(BufferedReader input) throws IOException {
		String[] word = ConfigEntry.readLineWords(input, 4);
		//System.err.print("Update: { ");
		//for (int i = 0; i < word.length; i++) {
		//	System.err.print("\"" + word[i] + "\", ");
		//}
		//System.err.println("}");
		if (word.length >= 1 && word[0].equals("set")) {
			inputSingleEntry(word, 1);
		} else if (word.length >= 1 && word[0].equals("pulse")) {
			// just ignore heartbeats
		} else {
			System.err.print("Unknown command: ");
			for (int i = 0; i < word.length; i++) {
				System.err.print("\"" + word[i] + "\"");
				if (i < word.length - 1) {
					System.err.print(", ");
				}
			}
			System.err.println();
		}
	}

	public void monitorInputForever() {
		try {
			InputStreamReader in = new InputStreamReader(this.in);
			BufferedReader input = new BufferedReader(in);
			//while (in.ready()) {
			//while (input.ready()) {
			while (true) {
				readUpdate(input);
			}
		} catch (IOException e) {
			System.err.println("exception " + e );
		}
	}

	//public friend istream& operator>>(istream& in, ConfigSet &configSet);
	//public friend ostream& operator<<(ostream& out, const ConfigSet &configSet);

}

